package com.example.canal.service;

import com.alibaba.fastjson.JSON;
import com.alibaba.otter.canal.client.CanalConnector;
import com.alibaba.otter.canal.protocol.Message;
import com.alibaba.otter.canal.protocol.CanalEntry.*;
import com.example.canal.config.RabbitMQConfig;
import lombok.extern.slf4j.Slf4j;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * 数据同步服务类
 * 
 * @author YourName
 * @since 2023-XX-XX
 */
@Slf4j
@Service
public class DataSyncService {
    
    @Autowired
    private CanalConnector canalConnector;
    
    @Autowired
    private RabbitTemplate rabbitTemplate;
    
    @Autowired
    private RedisTemplate<String, Object> redisTemplate;
    
    // 批量处理大小
    private static final int BATCH_SIZE = 1000;
    
    // 是否运行标志
    private volatile boolean running = false;
    
    // 数据处理线程
    private Thread workerThread;
    
    /**
     * 初始化方法
     */
    @PostConstruct
    public void init() {
        running = true;
        workerThread = new Thread(this::processData);
        workerThread.setName("Canal-Data-Sync-Thread");
        workerThread.start();
        log.info("数据同步服务已启动");
    }
    
    /**
     * 销毁方法
     */
    @PreDestroy
    public void destroy() {
        running = false;
        if (workerThread != null) {
            workerThread.interrupt();
        }
        log.info("数据同步服务已停止");
    }
    
    /**
     * 处理数据变更
     */
    private void processData() {
        while (running) {
            try {
                // 获取指定数量的数据
                Message message = canalConnector.getWithoutAck(BATCH_SIZE);
                
                // 获取批次ID
                long batchId = message.getId();
                
                if (batchId == -1 || message.getEntries().isEmpty()) {
                    // 没有数据时休眠一段时间
                    TimeUnit.MILLISECONDS.sleep(1000);
                    continue;
                }
                
                // 处理数据变更
                processEntries(message.getEntries());
                
                // 提交确认
                canalConnector.ack(batchId);
                
            } catch (Exception e) {
                log.error("处理数据变更时发生异常", e);
                try {
                    TimeUnit.SECONDS.sleep(5);
                } catch (InterruptedException ie) {
                    Thread.currentThread().interrupt();
                }
            }
        }
    }
    
    /**
     * 处理变更条目
     * 
     * @param entries 变更条目列表
     */
    private void processEntries(java.util.List<Entry> entries) {
        for (Entry entry : entries) {
            // 只处理行数据变更
            if (entry.getEntryType() == EntryType.ROWDATA) {
                try {
                    // 解析行数据变更
                    RowChange rowChange = RowChange.parseFrom(entry.getStoreValue());
                    
                    // 处理每一行数据
                    processRowData(entry.getHeader(), rowChange);
                } catch (Exception e) {
                    log.error("解析行数据变更时发生异常", e);
                }
            }
        }
    }
    
    /**
     * 处理行数据
     * 
     * @param header 头部信息
     * @param rowChange 行变更信息
     */
    private void processRowData(Header header, RowChange rowChange) {
        // 获取表名
        String tableName = header.getTableName();
        
        // 获取数据库名
        String databaseName = header.getSchemaName();
        
        // 获取操作类型
        EventType eventType = rowChange.getEventType();
        
        // 遍历每一行数据
        for (RowData rowData : rowChange.getRowDatasList()) {
            // 构造数据变更消息
            Map<String, Object> dataMap = new HashMap<>();
            dataMap.put("database", databaseName);
            dataMap.put("table", tableName);
            dataMap.put("type", eventType.toString());
            
            if (eventType == EventType.INSERT || eventType == EventType.UPDATE) {
                // 插入或更新操作，获取变更后的数据
                Map<String, Object> afterData = new HashMap<>();
                for (Column column : rowData.getAfterColumnsList()) {
                    afterData.put(column.getName(), column.getValue());
                }
                dataMap.put("data", afterData);
            } else if (eventType == EventType.DELETE) {
                // 删除操作，获取变更前的数据
                Map<String, Object> beforeData = new HashMap<>();
                for (Column column : rowData.getBeforeColumnsList()) {
                    beforeData.put(column.getName(), column.getValue());
                }
                dataMap.put("data", beforeData);
            }
            
            // 发送到RabbitMQ
            sendToRabbitMQ(dataMap);
            
            // 更新Redis缓存
            updateRedisCache(dataMap);
        }
    }
    
    /**
     * 发送到RabbitMQ
     * 
     * @param dataMap 数据映射
     */
    private void sendToRabbitMQ(Map<String, Object> dataMap) {
        try {
            String jsonData = JSON.toJSONString(dataMap);
            rabbitTemplate.convertAndSend(
                RabbitMQConfig.EXCHANGE_NAME,
                RabbitMQConfig.ROUTING_KEY,
                jsonData
            );
            log.info("数据已发送到RabbitMQ: {}", jsonData);
        } catch (Exception e) {
            log.error("发送数据到RabbitMQ时发生异常", e);
        }
    }
    
    /**
     * 更新Redis缓存
     * 
     * @param dataMap 数据映射
     */
    private void updateRedisCache(Map<String, Object> dataMap) {
        try {
            String table = (String) dataMap.get("table");
            String type = (String) dataMap.get("type");
            Map<String, Object> data = (Map<String, Object>) dataMap.get("data");
            
            if (data != null && !data.isEmpty()) {
                // 构造缓存键
                String cacheKey = "canal:" + table;
                
                // 根据操作类型更新缓存
                if ("INSERT".equals(type) || "UPDATE".equals(type)) {
                    // 对于插入和更新操作，将数据存入Redis
                    String id = (String) data.get("id");
                    if (id != null) {
                        String itemKey = cacheKey + ":" + id;
                        redisTemplate.opsForValue().set(itemKey, data, 30, TimeUnit.MINUTES);
                    }
                } else if ("DELETE".equals(type)) {
                    // 对于删除操作，从Redis中删除数据
                    String id = (String) data.get("id");
                    if (id != null) {
                        String itemKey = cacheKey + ":" + id;
                        redisTemplate.delete(itemKey);
                    }
                }
            }
        } catch (Exception e) {
            log.error("更新Redis缓存时发生异常", e);
        }
    }
}