package com.example.customer.service;

import dev.langchain4j.data.document.Document;
import dev.langchain4j.data.embedding.Embedding;
import dev.langchain4j.data.message.UserMessage;
import dev.langchain4j.model.chat.ChatModel;
import dev.langchain4j.model.embedding.EmbeddingModel;
import dev.langchain4j.rag.content.Content;
import dev.langchain4j.rag.content.retriever.ContentRetriever;
import dev.langchain4j.rag.content.retriever.EmbeddingStoreContentRetriever;
import dev.langchain4j.rag.query.Query;
import dev.langchain4j.rag.query.router.DefaultQueryRouter;
import dev.langchain4j.rag.query.router.QueryRouter;
import dev.langchain4j.service.AiServices;
import dev.langchain4j.service.SystemMessage;
import org.springframework.ai.vectorstore.SearchRequest;
import org.springframework.ai.vectorstore.VectorStore;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Map;

@Service
public class IntelligentCustomerService {
    
    private final ChatModel chatModel;
    private final EmbeddingModel embeddingModel;
    private final VectorStore vectorStore;
    
    public IntelligentCustomerService(ChatModel chatModel, 
                                    EmbeddingModel embeddingModel,
                                    VectorStore vectorStore) {
        this.chatModel = chatModel;
        this.embeddingModel = embeddingModel;
        this.vectorStore = vectorStore;
    }
    
    /**
     * 根据用户问题生成回答
     */
    public String getAnswer(String question) {
        // 使用向量存储检索相关信息
        List<Document> relevantDocuments = vectorStore.similaritySearch(
            SearchRequest.query(question)
                .withTopK(3)
                .withSimilarityThreshold(0.7)
        );
        
        // 构建提示词，包含检索到的上下文
        String context = relevantDocuments.stream()
            .map(Document::text)
            .reduce((s1, s2) -> s1 + "\n" + s2)
            .orElse("无相关上下文信息");
        
        String prompt = String.format(
            "基于以下上下文信息回答问题：\n\n上下文：%s\n\n问题：%s\n\n请根据上下文提供准确的回答，如果上下文信息不足，请说明信息不足。",
            context, question
        );
        
        // 使用Ollama模型生成回答
        return chatModel.generate(prompt);
    }
    
    /**
     * 向知识库添加文档
     */
    public void addDocument(String content, Map<String, Object> metadata) {
        Document document = Document.from(content).withMetadata(metadata);
        vectorStore.add(List.of(document));
    }
    
    /**
     * 从知识库删除文档
     */
    public void removeDocument(String id) {
        vectorStore.delete(List.of(id));
    }
}