package com.example.aviator.service;

import com.example.aviator.entity.Rule;
import com.example.aviator.engine.AviatorRuleEngine;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.HashMap;
import java.util.Map;

/**
 * 规则服务
 */
@Service
@Slf4j
public class RuleService {
    
    @Autowired
    private AviatorRuleEngine aviatorRuleEngine;
    
    /**
     * 执行营销优惠计算规则
     * 
     * @param ruleCode 规则编码
     * @param context 计算上下文
     * @return 计算结果
     */
    public Object executeMarketingRule(String ruleCode, Map<String, Object> context) {
        // 这里可以从数据库或缓存中获取规则表达式
        String ruleExpression = getRuleExpressionByCode(ruleCode);
        
        if (ruleExpression == null) {
            throw new RuntimeException("规则不存在: " + ruleCode);
        }
        
        log.info("执行营销规则: {}, 表达式: {}", ruleCode, ruleExpression);
        
        // 执行规则
        Object result = aviatorRuleEngine.executeRule(ruleExpression, context);
        
        log.info("规则执行结果: {} -> {}", ruleCode, result);
        return result;
    }
    
    /**
     * 根据订单金额和用户等级计算折扣
     * 
     * @param orderAmount 订单金额
     * @param userLevel 用户等级
     * @param productCategory 商品分类
     * @return 折扣率
     */
    public Double calculateDiscount(Double orderAmount, String userLevel, String productCategory) {
        Map<String, Object> context = new HashMap<>();
        context.put("orderAmount", orderAmount);
        context.put("userLevel", userLevel);
        context.put("productCategory", productCategory);
        
        // 根据不同用户等级和订单金额计算折扣
        String ruleExpression;
        if ("VIP".equals(userLevel)) {
            // VIP用户：订单金额>1000打8折，否则打9折
            ruleExpression = "orderAmount > 1000 ? 0.8 : 0.9";
        } else if ("GOLD".equals(userLevel)) {
            // 金牌用户：订单金额>500打8.5折，否则打9.5折
            ruleExpression = "orderAmount > 500 ? 0.85 : 0.95";
        } else {
            // 普通用户：订单金额>1000打9折，否则无折扣
            ruleExpression = "orderAmount > 1000 ? 0.9 : 1.0";
        }
        
        Object result = aviatorRuleEngine.executeRule(ruleExpression, context);
        return Double.valueOf(result.toString());
    }
    
    /**
     * 计算满减优惠
     * 
     * @param orderAmount 订单金额
     * @param couponType 优惠券类型
     * @return 优惠金额
     */
    public Double calculateCouponDiscount(Double orderAmount, String couponType) {
        Map<String, Object> context = new HashMap<>();
        context.put("orderAmount", orderAmount);
        context.put("couponType", couponType);
        
        String ruleExpression;
        if ("MANYIJIAN".equals(couponType)) {
            // 满减券：满200减20，满500减50
            ruleExpression = "orderAmount >= 500 ? 50 : (orderAmount >= 200 ? 20 : 0)";
        } else if ("ZHEKOU".equals(couponType)) {
            // 折扣券：满100打9折
            ruleExpression = "orderAmount >= 100 ? orderAmount * 0.1 : 0";
        } else {
            ruleExpression = "0";
        }
        
        Object result = aviatorRuleEngine.executeRule(ruleExpression, context);
        return Double.valueOf(result.toString());
    }
    
    /**
     * 计算会员等级升级条件
     * 
     * @param monthlyConsumption 月消费金额
     * @param totalConsumption 总消费金额
     * @param orderCount 订单数量
     * @return 是否满足升级条件
     */
    public Boolean checkUpgradeCondition(Double monthlyConsumption, Double totalConsumption, Integer orderCount) {
        Map<String, Object> context = new HashMap<>();
        context.put("monthlyConsumption", monthlyConsumption);
        context.put("totalConsumption", totalConsumption);
        context.put("orderCount", orderCount);
        
        // 升级规则：月消费>1000 或 (总消费>5000 且 订单数>10)
        String ruleExpression = "(monthlyConsumption > 1000) or (totalConsumption > 5000 and orderCount > 10)";
        
        Object result = aviatorRuleEngine.executeRule(ruleExpression, context);
        return Boolean.valueOf(result.toString());
    }
    
    /**
     * 根据规则编码获取规则表达式（模拟从数据库获取）
     * 
     * @param ruleCode 规则编码
     * @return 规则表达式
     */
    private String getRuleExpressionByCode(String ruleCode) {
        // 这里实际项目中应该从数据库查询
        switch (ruleCode) {
            case "DISCOUNT_RULE_VIP":
                return "orderAmount > 1000 ? 0.8 : 0.9";
            case "DISCOUNT_RULE_GOLD":
                return "orderAmount > 500 ? 0.85 : 0.95";
            case "DISCOUNT_RULE_NORMAL":
                return "orderAmount > 1000 ? 0.9 : 1.0";
            case "COUPON_MANYIJIAN":
                return "orderAmount >= 500 ? 50 : (orderAmount >= 200 ? 20 : 0)";
            case "COUPON_ZHEKOU":
                return "orderAmount >= 100 ? orderAmount * 0.1 : 0";
            case "UPGRADE_RULE":
                return "(monthlyConsumption > 1000) or (totalConsumption > 5000 and orderCount > 10)";
            default:
                return null;
        }
    }
}