package com.example.aviator.engine;

import com.googlecode.aviator.AviatorEvaluator;
import com.googlecode.aviator.Expression;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Aviator规则引擎
 */
@Component
@Slf4j
public class AviatorRuleEngine {
    
    /**
     * 缓存编译后的表达式，提高执行效率
     */
    private final Map<String, Expression> expressionCache = new ConcurrentHashMap<>();
    
    /**
     * 执行规则表达式
     * 
     * @param expression 规则表达式
     * @param env 执行环境变量
     * @return 执行结果
     */
    public Object executeRule(String expression, Map<String, Object> env) {
        try {
            // 检查缓存中是否存在编译后的表达式
            Expression compiledExpression = expressionCache.get(expression);
            if (compiledExpression == null) {
                // 编译表达式并缓存
                compiledExpression = AviatorEvaluator.compile(expression, true);
                expressionCache.put(expression, compiledExpression);
                log.debug("缓存表达式: {}", expression);
            }
            
            // 执行表达式
            Object result = compiledExpression.execute(env);
            log.debug("表达式执行结果: {} = {}", expression, result);
            return result;
        } catch (Exception e) {
            log.error("执行规则表达式失败: {}", expression, e);
            throw new RuntimeException("规则执行失败: " + e.getMessage(), e);
        }
    }
    
    /**
     * 验证表达式语法
     * 
     * @param expression 表达式
     * @return 是否有效
     */
    public boolean validateExpression(String expression) {
        try {
            AviatorEvaluator.compile(expression, true);
            return true;
        } catch (Exception e) {
            log.error("表达式语法错误: {}", expression, e);
            return false;
        }
    }
    
    /**
     * 清除表达式缓存
     */
    public void clearCache() {
        expressionCache.clear();
        log.info("清空表达式缓存");
    }
    
    /**
     * 移除指定表达式的缓存
     * 
     * @param expression 表达式
     */
    public void removeFromCache(String expression) {
        expressionCache.remove(expression);
        log.debug("移除表达式缓存: {}", expression);
    }
}