package com.example.aviator.controller;

import com.example.aviator.service.RuleService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.HashMap;
import java.util.Map;

/**
 * 营销规则控制器
 */
@RestController
@RequestMapping("/api/marketing")
@Slf4j
public class MarketingRuleController {
    
    @Autowired
    private RuleService ruleService;
    
    /**
     * 计算订单折扣
     */
    @PostMapping("/calculate-discount")
    public ResponseEntity<Map<String, Object>> calculateDiscount(@RequestBody Map<String, Object> request) {
        try {
            Double orderAmount = Double.valueOf(request.get("orderAmount").toString());
            String userLevel = request.get("userLevel").toString();
            String productCategory = request.get("productCategory").toString();
            
            Double discountRate = ruleService.calculateDiscount(orderAmount, userLevel, productCategory);
            
            Map<String, Object> result = new HashMap<>();
            result.put("originalAmount", orderAmount);
            result.put("discountRate", discountRate);
            result.put("discountedAmount", orderAmount * discountRate);
            result.put("savedAmount", orderAmount * (1 - discountRate));
            result.put("userLevel", userLevel);
            
            log.info("订单折扣计算完成: {}", result);
            return ResponseEntity.ok(result);
        } catch (Exception e) {
            log.error("计算订单折扣失败", e);
            Map<String, Object> error = new HashMap<>();
            error.put("error", e.getMessage());
            return ResponseEntity.badRequest().body(error);
        }
    }
    
    /**
     * 计算优惠券折扣
     */
    @PostMapping("/calculate-coupon")
    public ResponseEntity<Map<String, Object>> calculateCoupon(@RequestBody Map<String, Object> request) {
        try {
            Double orderAmount = Double.valueOf(request.get("orderAmount").toString());
            String couponType = request.get("couponType").toString();
            
            Double discountAmount = ruleService.calculateCouponDiscount(orderAmount, couponType);
            
            Map<String, Object> result = new HashMap<>();
            result.put("originalAmount", orderAmount);
            result.put("discountAmount", discountAmount);
            result.put("finalAmount", orderAmount - discountAmount);
            result.put("couponType", couponType);
            
            log.info("优惠券计算完成: {}", result);
            return ResponseEntity.ok(result);
        } catch (Exception e) {
            log.error("计算优惠券失败", e);
            Map<String, Object> error = new HashMap<>();
            error.put("error", e.getMessage());
            return ResponseEntity.badRequest().body(error);
        }
    }
    
    /**
     * 检查会员升级条件
     */
    @PostMapping("/check-upgrade")
    public ResponseEntity<Map<String, Object>> checkUpgrade(@RequestBody Map<String, Object> request) {
        try {
            Double monthlyConsumption = Double.valueOf(request.get("monthlyConsumption").toString());
            Double totalConsumption = Double.valueOf(request.get("totalConsumption").toString());
            Integer orderCount = Integer.valueOf(request.get("orderCount").toString());
            
            Boolean canUpgrade = ruleService.checkUpgradeCondition(monthlyConsumption, totalConsumption, orderCount);
            
            Map<String, Object> result = new HashMap<>();
            result.put("monthlyConsumption", monthlyConsumption);
            result.put("totalConsumption", totalConsumption);
            result.put("orderCount", orderCount);
            result.put("canUpgrade", canUpgrade);
            
            log.info("会员升级检查完成: {}", result);
            return ResponseEntity.ok(result);
        } catch (Exception e) {
            log.error("检查会员升级失败", e);
            Map<String, Object> error = new HashMap<>();
            error.put("error", e.getMessage());
            return ResponseEntity.badRequest().body(error);
        }
    }
    
    /**
     * 执行自定义规则
     */
    @PostMapping("/execute-rule")
    public ResponseEntity<Map<String, Object>> executeRule(@RequestBody Map<String, Object> request) {
        try {
            String ruleCode = request.get("ruleCode").toString();
            @SuppressWarnings("unchecked")
            Map<String, Object> context = (Map<String, Object>) request.get("context");
            
            Object result = ruleService.executeMarketingRule(ruleCode, context);
            
            Map<String, Object> response = new HashMap<>();
            response.put("ruleCode", ruleCode);
            response.put("context", context);
            response.put("result", result);
            
            log.info("自定义规则执行完成: {}", response);
            return ResponseEntity.ok(response);
        } catch (Exception e) {
            log.error("执行自定义规则失败", e);
            Map<String, Object> error = new HashMap<>();
            error.put("error", e.getMessage());
            return ResponseEntity.badRequest().body(error);
        }
    }
}