package com.example.datamasking.util;

import com.example.datamasking.annotation.SensitiveData;

/**
 * 敏感数据处理器
 * 提供各种脱敏方法
 */
public class SensitiveDataHandler {

    /**
     * 根据注解配置进行脱敏处理
     *
     * @param value 原始值
     * @param sensitiveData 脱敏注解
     * @return 脱敏后的值
     */
    public static String mask(String value, SensitiveData sensitiveData) {
        if (value == null || value.isEmpty() || !sensitiveData.enabled()) {
            return value;
        }

        switch (sensitiveData.type()) {
            case CHINESE_NAME:
                return maskChineseName(value);
            case ID_CARD:
                return maskIdCard(value);
            case PHONE:
                return maskPhone(value);
            case FIXED_PHONE:
                return maskFixedPhone(value);
            case EMAIL:
                return maskEmail(value);
            case BANK_CARD:
                return maskBankCard(value);
            case ADDRESS:
                return maskAddress(value);
            case CUSTOM:
                return maskCustom(value, sensitiveData.prefixKeep(), sensitiveData.suffixKeep(), sensitiveData.maskChar());
            default:
                return value;
        }
    }

    /**
     * 中文姓名脱敏（保留姓氏，其余用*代替）
     * 示例：张三 -> 张*
     * 示例：李小明 -> 李**
     */
    public static String maskChineseName(String name) {
        if (name == null || name.length() == 0) {
            return name;
        }

        if (name.length() == 1) {
            return name;
        } else if (name.length() == 2) {
            return name.charAt(0) + "*";
        } else {
            return name.substring(0, 1) + "*" + name.substring(2);
        }
    }

    /**
     * 身份证号码脱敏（保留前3位和后3位）
     * 示例：110101199003071234 -> 110****1234
     */
    public static String maskIdCard(String idCard) {
        if (idCard == null || idCard.length() < 8) {
            return idCard;
        }

        int length = idCard.length();
        String prefix = idCard.substring(0, 3);
        String suffix = idCard.substring(length - 3);
        String mask = maskString("", length - 6, '*');

        return prefix + mask + suffix;
    }

    /**
     * 手机号码脱敏（保留前3位和后4位）
     * 示例：13812345678 -> 138****5678
     */
    public static String maskPhone(String phone) {
        if (phone == null || phone.length() != 11) {
            return phone;
        }

        return phone.substring(0, 3) + "****" + phone.substring(7);
    }

    /**
     * 固定电话脱敏（保留区号和后4位）
     * 示例：010-12345678 -> 010****5678
     */
    public static String maskFixedPhone(String phone) {
        if (phone == null || phone.length() < 7) {
            return phone;
        }

        int lastFourIndex = Math.max(0, phone.length() - 4);
        String prefix = phone.substring(0, Math.min(4, phone.length() - 4));
        String suffix = phone.substring(lastFourIndex);
        String mask = maskString("", Math.max(0, phone.length() - prefix.length() - suffix.length), '*');

        return prefix + mask + suffix;
    }

    /**
     * 电子邮件脱敏（保留用户名首尾各1位）
     * 示例：example@domain.com -> e***e@domain.com
     */
    public static String maskEmail(String email) {
        if (email == null || !email.contains("@")) {
            return email;
        }

        int atIndex = email.indexOf('@');
        String username = email.substring(0, atIndex);
        String domain = email.substring(atIndex);

        if (username.length() <= 2) {
            return username.charAt(0) + "*" + domain;
        } else {
            String firstChar = String.valueOf(username.charAt(0));
            String lastChar = String.valueOf(username.charAt(username.length() - 1));
            String mask = maskString("", username.length() - 2, '*');
            return firstChar + mask + lastChar + domain;
        }
    }

    /**
     * 银行卡号脱敏（保留前6位和后4位）
     * 示例：6222021234567890123 -> 622202******90123
     */
    public static String maskBankCard(String cardNo) {
        if (cardNo == null || cardNo.length() < 10) {
            return cardNo;
        }

        int length = cardNo.length();
        String prefix = cardNo.substring(0, 6);
        String suffix = cardNo.substring(length - 4);
        String mask = maskString("", length - 10, '*');

        return prefix + mask + suffix;
    }

    /**
     * 地址脱敏（保留前6位字符）
     * 示例：北京市朝阳区某某街道123号 -> 北京市朝阳区******
     */
    public static String maskAddress(String address) {
        if (address == null || address.length() <= 6) {
            return address;
        }

        return address.substring(0, 6) + "******";
    }

    /**
     * 自定义脱敏规则
     * @param value 原始值
     * @param prefixKeep 前缀保留长度
     * @param suffixKeep 后缀保留长度
     * @param maskChar 替换字符
     * @return 脱敏后的值
     */
    public static String maskCustom(String value, int prefixKeep, int suffixKeep, String maskChar) {
        if (value == null) {
            return value;
        }

        int length = value.length();
        if (length <= prefixKeep + suffixKeep) {
            return value;
        }

        String prefix = value.substring(0, prefixKeep);
        String suffix = value.length() > suffixKeep ? value.substring(length - suffixKeep) : "";
        String mask = maskString("", length - prefixKeep - suffixKeep, maskChar.charAt(0));

        return prefix + mask + suffix;
    }

    /**
     * 创建指定长度的掩码字符串
     *
     * @param str 原字符串（用于拼接）
     * @param length 掩码长度
     * @param maskChar 掩码字符
     * @return 掩码字符串
     */
    private static String maskString(String str, int length, char maskChar) {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < length; i++) {
            sb.append(maskChar);
        }
        return sb.toString();
    }
}