package com.example.datamasking.aspect;

import com.example.datamasking.annotation.SensitiveData;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.core.annotation.Order;
import org.springframework.stereotype.Component;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.Collection;
import java.util.Map;

/**
 * 敏感数据脱敏切面
 * 拦截返回数据，对标注了@SensitiveData注解的字段进行脱敏处理
 */
@Aspect
@Order(1) // 设置优先级，确保在其他切面之前执行
@Component
public class SensitiveDataAspect {

    /**
     * 环绕通知，处理方法返回值的脱敏
     */
    @Around("execution(* *(..)) && (@annotation(org.springframework.web.bind.annotation.GetMapping) || " +
            "@annotation(org.springframework.web.bind.annotation.PostMapping) || " +
            "@annotation(org.springframework.web.bind.annotation.PutMapping) || " +
            "@annotation(org.springframework.web.bind.annotation.DeleteMapping) || " +
            "@annotation(org.springframework.web.bind.annotation.RequestMapping))")
    public Object around(ProceedingJoinPoint point) throws Throwable {
        // 执行原方法
        Object result = point.proceed();
        
        // 对返回结果进行脱敏处理
        if (result != null) {
            result = processSensitiveData(result);
        }
        
        return result;
    }

    /**
     * 处理敏感数据
     * @param obj 待处理对象
     * @return 处理后的对象
     */
    private Object processSensitiveData(Object obj) {
        if (obj == null) {
            return obj;
        }

        if (obj instanceof String) {
            // 如果是String类型，直接返回（因为String是不可变的，且通常不会在返回对象中直接使用String作为复杂数据）
            return obj;
        } else if (obj instanceof Collection) {
            // 处理集合类型
            Collection<?> collection = (Collection<?>) obj;
            for (Object item : collection) {
                processSensitiveData(item);
            }
        } else if (obj instanceof Map) {
            // 处理Map类型
            Map<?, ?> map = (Map<?, ?>) obj;
            for (Object value : map.values()) {
                processSensitiveData(value);
            }
        } else {
            // 处理普通对象
            processObjectSensitiveData(obj);
        }

        return obj;
    }

    /**
     * 处理对象中的敏感数据
     * @param obj 待处理对象
     */
    private void processObjectSensitiveData(Object obj) {
        if (obj == null) {
            return;
        }

        Class<?> clazz = obj.getClass();
        
        // 处理当前类及其父类的所有字段
        while (clazz != null && !clazz.equals(Object.class)) {
            Field[] fields = clazz.getDeclaredFields();
            for (Field field : fields) {
                try {
                    field.setAccessible(true);
                    Object value = field.get(obj);

                    // 检查字段是否有@SensitiveData注解
                    if (field.isAnnotationPresent(SensitiveData.class)) {
                        SensitiveData sensitiveData = field.getAnnotation(SensitiveData.class);
                        
                        if (value instanceof String) {
                            String originalValue = (String) value;
                            String maskedValue = com.example.datamasking.util.SensitiveDataHandler.mask(originalValue, sensitiveData);
                            field.set(obj, maskedValue);
                        } else if (value instanceof Collection) {
                            // 递归处理集合中的元素
                            Collection<?> collection = (Collection<?>) value;
                            for (Object item : collection) {
                                processSensitiveData(item);
                            }
                        } else if (value instanceof Map) {
                            // 递归处理Map中的值
                            Map<?, ?> map = (Map<?, ?>) value;
                            for (Object mapValue : map.values()) {
                                processSensitiveData(mapValue);
                            }
                        } else if (value != null) {
                            // 递归处理嵌套对象
                            processSensitiveData(value);
                        }
                    } else {
                        // 如果字段没有@SensitiveData注解，但类型是复杂对象，继续递归处理
                        if (value != null && !isPrimitiveOrWrapper(value.getClass()) && !value.getClass().isEnum()) {
                            processSensitiveData(value);
                        }
                    }
                } catch (IllegalAccessException e) {
                    // 记录日志或处理异常
                    e.printStackTrace();
                }
            }
            clazz = clazz.getSuperclass();
        }
    }

    /**
     * 判断是否为基本类型或包装类型
     * @param clazz 类型
     * @return 是否为基本类型或包装类型
     */
    private boolean isPrimitiveOrWrapper(Class<?> clazz) {
        return clazz.isPrimitive() ||
                clazz.equals(String.class) ||
                clazz.equals(Boolean.class) ||
                clazz.equals(Character.class) ||
                clazz.equals(Byte.class) ||
                clazz.equals(Short.class) ||
                clazz.equals(Integer.class) ||
                clazz.equals(Long.class) ||
                clazz.equals(Float.class) ||
                clazz.equals(Double.class) ||
                clazz.equals(Void.class);
    }
}